<?php

/**
 * LibreDTE
 * Copyright (C) SASCO SpA (https://sasco.cl)
 *
 * Este programa es software libre: usted puede redistribuirlo y/o
 * modificarlo bajo los términos de la Licencia Pública General GNU
 * publicada por la Fundación para el Software Libre, ya sea la versión
 * 3 de la Licencia, o (a su elección) cualquier versión posterior de la
 * misma.
 *
 * Este programa se distribuye con la esperanza de que sea útil, pero
 * SIN GARANTÍA ALGUNA; ni siquiera la garantía implícita
 * MERCANTIL o de APTITUD PARA UN PROPÓSITO DETERMINADO.
 * Consulte los detalles de la Licencia Pública General GNU para obtener
 * una información más detallada.
 *
 * Debería haber recibido una copia de la Licencia Pública General GNU
 * junto a este programa.
 * En caso contrario, consulte <http://www.gnu.org/licenses/gpl.html>.
 */


/**
 * Clase para trabajar con firma electrónica, permite firmar y verificar firmas.
 * Provee los métodos: sign(), verify(), signXML() y verifyXML()
 * @author Esteban De La Fuente Rubio, DeLaF (esteban[at]sasco.cl)
 * @version 2015-08-24
 */
class FirmaElectronica
{

    private $config; ///< Configuración de la firma electrónica
    private $certs; ///< Certificados digitales de la firma

    /**
     * Constructor para la clase: crea configuración y carga certificado digital
     *
     * Si se desea pasar una configuración específica para la firma electrónica
     * se debe hacer a través de un arreglo con los índices file y pass, donde
     * file es la ruta hacia el archivo .p12 que contiene tanto la clave privada
     * como la pública y pass es la contraseña para abrir dicho archivo.
     * Ejemplo:
     *
     * \code{.php}
     *   $firma_config = ['file'=>'/ruta/al/certificado.p12', 'pass'=>'contraseña'];
     *   $firma = new \sasco\LibreDTE\FirmaElectronica($firma_config);
     * \endcode
     *
     * También se permite que en vez de pasar la ruta al certificado p12 se pase
     * el contenido del certificado, esto servirá por ejemplo si los datos del
     * archivo están almacenados en una base de datos. Ejemplo:
     *
     * \code{.php}
     *   $firma_config = ['data'=>file_get_contents('/ruta/al/certificado.p12'), 'pass'=>'contraseña'];
     *   $firma = new \sasco\LibreDTE\FirmaElectronica($firma_config);
     * \endcode
     *
     * @param config Configuración para la clase, si no se especifica se tratará de determinar
     * @author Esteban De La Fuente Rubio, DeLaF (esteban[at]sasco.cl)
     * @version 2015-08-25
     */
   public function __construct(array $config)
    {
        // crear configuración
       /* if (!$config) {
            if (class_exists('\sowerphp\core\Configure')) {
                $config = (array)\sowerphp\core\Configure::read('firma_electronica.default');
            } else {
                $config = [];
            }
        } */
        $this->config = array_merge(array(
            'file' => '',
            'pass' => '',
            'wordwrap' => 64,
        ), $config);
       
        
        // cargar firma electrónica desde el contenido del archivo .p12 si no
        // se pasaron como datos del arreglo de configuración
        if (empty($this->config['data'])) {
            if (is_readable($this->config['file'])) {
                $this->config['data'] = file_get_contents($this->config['file']);
            } else {
                $this->error('Archivo de la firma electrónica '.basename($this->config['file']).' no puede ser leído');
            }
        }
        // leer datos de la firma electrónica
        
         
        if (openssl_pkcs12_read($this->config['data'], $this->certs, $this->config['pass'])===false) {
            $this->error('No fue posible leer los datos de la firma electrónica (verificar la contraseña)');
        }
        // quitar datos del contenido del archivo de la firma
        unset($this->config['data']);
    }

    /**
     * Método para generar un error usando una excepción de SowerPHP o terminar
     * el script si no se está usando el framework
     * @param msg Mensaje del error
     * @author Esteban De La Fuente Rubio, DeLaF (esteban[at]sasco.cl)
     * @version 2014-12-06
     */
    private function error($msg)
    {
       /* if (class_exists('\sowerphp\core\Exception')) {
            throw new \sowerphp\core\Exception($msg);
        } else { */
            die($msg);
       // }
    }

    /**
     * Método que agrega el inicio y fin de un certificado (clave pública)
     * @param cert Certificado que se desea normalizar
     * @return Certificado con el inicio y fin correspondiente
     * @author Esteban De La Fuente Rubio, DeLaF (esteban[at]sasco.cl)
     * @version 2015-08-20
     */
    private function normalizeCert($cert)
    {
        if (strpos($cert, '-----BEGIN CERTIFICATE-----')===false) {
            $body = trim($cert);
            $cert = '-----BEGIN CERTIFICATE-----'."\n";
            $cert .= wordwrap($body, $this->config['wordwrap'], "\n", true)."\n";
            $cert .= '-----END CERTIFICATE-----'."\n";
        }
        return $cert;
    }

    /**
     * Método que obtiene el módulo de la clave privada
     * @return Módulo en base64
     * @author Esteban De La Fuente Rubio, DeLaF (esteban[at]sasco.cl)
     * @version 2014-12-07
     */
    public function getModulus()
    {
    	
		
		$resource = openssl_pkey_get_private($this->certs['pkey']);
        $details = openssl_pkey_get_details($resource);
        return wordwrap(base64_encode($details['rsa']['n']), $this->config['wordwrap'], "\n", true);
    }

    /**
     * Método que obtiene el exponente público de la clave privada
     * @return Exponente público en base64
     * @author Esteban De La Fuente Rubio, DeLaF (esteban[at]sasco.cl)
     * @version 2014-12-06
     */
    public function getExponent()
    {
        $details = openssl_pkey_get_details(openssl_pkey_get_private($this->certs['pkey']));
        return wordwrap(base64_encode($details['rsa']['e']), $this->config['wordwrap'], "\n", true);
    }

    /**
     * Método que entrega el certificado de la firma
     * @return Contenido del certificado, clave pública del certificado digital, en base64
     * @author Esteban De La Fuente Rubio, DeLaF (esteban[at]sasco.cl)
     * @version 2015-08-24
     */
    public function getCertificate($clean = false)
    {
        if ($clean) {
            return trim(str_replace(array('-----BEGIN CERTIFICATE-----', '-----END CERTIFICATE-----'),'',$this->certs['cert'] ));
        } else {
            return $this->certs['cert'];
        }
    }

    /**
     * Método que entrega la clave privada de la firma
     * @return Contenido de la clave privada del certificado digital en base64
     * @author Esteban De La Fuente Rubio, DeLaF (esteban[at]sasco.cl)
     * @version 2015-08-24
     */
    public function getPrivateKey($clean = false)
    {
        if ($clean) {
            return trim(str_replace(array('-----BEGIN PRIVATE KEY-----', '-----END PRIVATE KEY-----'),'',$this->certs['pkey']
            ));
        } else {
            return $this->certs['pkey'];
        }
    }

    /**
     * Método para realizar la firma de datos
     * @param data Datos que se desean firmar
     * @param signature_alg Algoritmo que se utilizará para firmar (por defect SHA1)
     * @return Firma digital de los datos en base64 o =false si no se pudo firmar
     * @author Esteban De La Fuente Rubio, DeLaF (esteban[at]sasco.cl)
     * @version 2014-12-08
     */
    public function sign($data, $signature_alg = OPENSSL_ALGO_SHA1)
    {
        $signature = null;
        if (openssl_sign($data, $signature, $this->certs['pkey'], $signature_alg)==false)
            return false;
        return base64_encode($signature);
    }

    /**
     * Método que verifica la firma digital de datos
     * @param data Datos que se desean verificar
     * @param signature Firma digital de los datos en base64
     * @param pub_key Certificado digital, clave pública, de la firma
     * @param signature_alg Algoritmo que se usó para firmar (por defect SHA1)
     * @return =true si la firma está ok, =false si está mal o no se pudo determinar
     * @author Esteban De La Fuente Rubio, DeLaF (esteban[at]sasco.cl)
     * @version 2014-12-08
     */
    public function verify($data, $signature, $pub_key = null, $signature_alg = OPENSSL_ALGO_SHA1)
    {
        if ($pub_key === null)
            $pub_key = $this->certs['cert'];
        $pub_key = $this->normalizeCert($pub_key);
        return openssl_verify($data, base64_decode($signature), $pub_key, $signature_alg) == 1 ? true : false;
    }

    /**
     * Método que firma un XML utilizando RSA y SHA1
     *
     * Referencia: http://www.di-mgt.com.au/xmldsig2.html
     *
     * @param xml Datos XML que se desean firmar
     * @param reference Referencia a la que hace la firma
     * @return XML firmado o =false si no se pudo fimar
     * @author Esteban De La Fuente Rubio, DeLaF (esteban[at]sasco.cl)
     * @version 2015-08-20
     */
    public function signXML($xml, $reference = '')
    {
    
    	
        $var_retorno = array();
        $doc = new DomDocument;
        $doc->loadXML($xml);
        $dom = $doc->documentElement;
        // crear nodo para la firma
        $xml_x = new XML;
        $arreglo_Edu = array(
            'Signature' => array(
                '@attributes' => array(
                    'xmlns' => 'http://www.w3.org/2000/09/xmldsig#',
                ),
                'SignedInfo' => array(
                    '@attributes' => array(
                        'xmlns' => 'http://www.w3.org/2000/09/xmldsig#',
                    ),
                    'CanonicalizationMethod' => array(
                        '@attributes' => array(
                            'Algorithm' => 'http://www.w3.org/TR/2001/REC-xml-c14n-20010315',
                        ),
                    ),
                    'SignatureMethod' => array(
                        '@attributes' => array(
                            'Algorithm' => 'http://www.w3.org/2000/09/xmldsig#rsa-sha1',
                        ),
                    ),
                    'Reference' => array(
                        '@attributes' => array(
                            'URI' => $reference,
                        ),
                        'Transforms' => array(
                            'Transform' => array(
                                '@attributes' => array(
                                    'Algorithm' => 'http://www.w3.org/2000/09/xmldsig#enveloped-signature',
                                ),
                            ),
                        ),
                        'DigestMethod' => array(
                            '@attributes' => array(
                                'Algorithm' => 'http://www.w3.org/2000/09/xmldsig#sha1',
                            ),
                        ),
                        'DigestValue' => null,
                    ),
                ),
                'SignatureValue' => null,
                'KeyInfo' => array(
                    'KeyValue' => array(
                        'RSAKeyValue' => array(
                            'Modulus' => null,
                            'Exponent' => null,
                        ),
                    ),
                    'X509Data' => array(
                        'X509Certificate' => null,
                    ),
                ),
            ),
        );
        
        
        
        $Signature = $doc->importNode($xml_x->generate($arreglo_Edu)->documentElement, true);
        
        
        
        
        // calcular DigestValue y SignatureValue
        $digest = base64_encode(sha1($dom->C14N(), true));
        
        
        $var_retorno['DigestValue'] = $digest;
        $Signature->getElementsByTagName('DigestValue')->item(0)->nodeValue = $digest;
        $SignedInfo = $Signature->getElementsByTagName('SignedInfo')->item(0);
        
        $firma = $this->sign($doc->saveHTML($SignedInfo));
       
        if (!$firma)
            return false;
        $signature = wordwrap($firma, $this->config['wordwrap'], "\n", true);
        
        
        $var_retorno['signature'] = $signature;

        // reemplazar valores en la firma de
        $Signature->getElementsByTagName('SignatureValue')->item(0)->nodeValue = $signature;
        
        $var_retorno['Modulus'] = $this->getModulus();
        $var_retorno['Exponent'] = $this->getExponent();
        $var_retorno['X509Certificate'] = $this->getCertificate(true);
       $Signature->getElementsByTagName('Modulus')->item(0)->nodeValue = $this->getModulus();
        
    
        
        $Signature->getElementsByTagName('Exponent')->item(0)->nodeValue = $this->getExponent();
        
        $Signature->getElementsByTagName('X509Certificate')->item(0)->nodeValue = $this->getCertificate(true);
        $dom->appendChild($Signature);
        $doc->saveXML();
        $doc->save("xml_firma/firma.xml");

       return $doc->saveXML();
    }

    /**
     * Método que verifica la validez de la firma de un XML utilizando RSA y SHA1
     * @param xml_data Archivo XML que se desea validar
     * @return =true si la firma del documento XML es válida o =false si no lo es
     * @author Esteban De La Fuente Rubio, DeLaF (esteban[at]sasco.cl)
     * @version 2015-08-20
     */
    public function verifyXML($xml_data)
    {
        $doc = new DomDocument;
        $doc->loadXML($xml_data);
        $dom = $doc->documentElement;
        // preparar datos que se verificarán
        $SignaturesElements = $dom->getElementsByTagName('Signature');
        $Signature = $dom->removeChild($SignaturesElements->item($SignaturesElements->length-1));
        $SignedInfo = $Signature->getElementsByTagName('SignedInfo');
        
        
        $SignedInfo->setAttribute('xmlns', $Signature->getAttribute('xmlns'));
        $signed_info = $doc->saveHTML($SignedInfo);
        $signature_p = $Signature->getElementsByTagName('SignatureValue');
        foreach ($signature_p as $valor) {
    		$signature =  $valor->nodeValue;
    		break;
			}
        
        $pub_key_p = $Signature->getElementsByTagName('X509Certificate');
        foreach ($pub_key_p as $valor) {
    		$pub_key =  $valor->nodeValue;
    		break;
			}
        // verificar firma
        if (!$this->verify($signed_info, $signature, $pub_key))
            return false;
        // verificar digest
        $digest_original_p = $Signature->getElementsByTagName('DigestValue');
        foreach ($digest_original_p as $valor) {
    		$digest_original =  $valor->nodeValue;
    		break;
			}
        $digest_calculado = base64_encode(sha1($dom->C14N(), true));
        return $digest_original == $digest_calculado;
    }

}
